<?php

namespace App\Http\Controllers;

use App\Http\Requests\CreateProposalSnippetRequest;
use App\Http\Requests\ProposalSnippetRequest;
use App\Http\Requests\UpdateProposalSnippetRequest;
use App\Models\Invoice;
use App\Models\ProposalSnippet;
use App\Models\ProposalCategory;
use App\Ninja\Datatables\ProposalSnippetDatatable;
use App\Ninja\Repositories\ProposalSnippetRepository;
use App\Services\ProposalSnippetService;
use Auth;
use Input;
use Session;
use View;

class ProposalSnippetController extends BaseController
{
    protected $proposalSnippetRepo;
    protected $proposalSnippetService;
    protected $entityType = ENTITY_PROPOSAL_SNIPPET;

    public function __construct(ProposalSnippetRepository $proposalSnippetRepo, ProposalSnippetService $proposalSnippetService)
    {
        $this->proposalSnippetRepo = $proposalSnippetRepo;
        $this->proposalSnippetService = $proposalSnippetService;
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index()
    {
        return View::make('list_wrapper', [
            'entityType' => ENTITY_PROPOSAL_SNIPPET,
            'datatable' => new ProposalSnippetDatatable(),
            'title' => trans('texts.proposal_snippets'),
        ]);
    }

    public function getDatatable($expensePublicId = null)
    {
        $search = Input::get('sSearch');
        $userId = Auth::user()->filterId();

        return $this->proposalSnippetService->getDatatable($search, $userId);
    }

    public function create(ProposalSnippetRequest $request)
    {
        $data = [
            'account' => auth()->user()->account,
            'snippet' => null,
            'method' => 'POST',
            'url' => 'proposals/snippets',
            'title' => trans('texts.new_proposal_snippet'),
            'categories' => ProposalCategory::scope()->orderBy('name')->get(),
            'categoryPublicId' => 0,
            'icons' => $this->getIcons(),
        ];

        return View::make('proposals/snippets/edit', $data);
    }

    public function show($publicId)
    {
        Session::reflash();

        return redirect("proposals/snippets/$publicId/edit");
    }

    public function edit(ProposalSnippetRequest $request)
    {
        $proposalSnippet = $request->entity();

        $data = [
            'account' => auth()->user()->account,
            'snippet' => $proposalSnippet,
            'entity' => $proposalSnippet,
            'method' => 'PUT',
            'url' => 'proposals/snippets/' . $proposalSnippet->public_id,
            'title' => trans('texts.edit_proposal_snippet'),
            'categories' => ProposalCategory::scope()->withActiveOrSelected($proposalSnippet->proposal_category_id)->orderBy('name')->get(),
            'categoryPublicId' => $proposalSnippet->proposal_category ? $proposalSnippet->proposal_category->public_id : null,
            'icons' => $this->getIcons(),
        ];

        return View::make('proposals/snippets.edit', $data);
    }

    public function store(CreateProposalSnippetRequest $request)
    {
        $proposalSnippet = $this->proposalSnippetService->save($request->input());

        Session::flash('message', trans('texts.created_proposal_snippet'));

        return redirect()->to($proposalSnippet->getRoute());
    }

    public function update(UpdateProposalSnippetRequest $request)
    {
        $proposalSnippet = $this->proposalSnippetService->save($request->input(), $request->entity());

        Session::flash('message', trans('texts.updated_proposal_snippet'));

        $action = Input::get('action');
        if (in_array($action, ['archive', 'delete', 'restore'])) {
            return self::bulk();
        }

        return redirect()->to($proposalSnippet->getRoute());
    }

    public function bulk()
    {
        $action = Input::get('action');
        $ids = Input::get('public_id') ? Input::get('public_id') : Input::get('ids');

        $count = $this->proposalSnippetService->bulk($ids, $action);

        if ($count > 0) {
            $field = $count == 1 ? "{$action}d_proposal_snippet" : "{$action}d_proposal_snippets";
            $message = trans("texts.$field", ['count' => $count]);
            Session::flash('message', $message);
        }

        return redirect()->to('/proposals/snippets');
    }

    private function getIcons() {
        $data = [];
        $icons = [
            ['name'=>'glass','code'=>'f000'],
            ['name'=>'music','code'=>'f001'],
            ['name'=>'search','code'=>'f002'],
            ['name'=>'envelope-o','code'=>'f003'],
            ['name'=>'heart','code'=>'f004'],
            ['name'=>'star','code'=>'f005'],
            ['name'=>'star-o','code'=>'f006'],
            ['name'=>'user','code'=>'f007'],
            ['name'=>'film','code'=>'f008'],
            ['name'=>'th-large','code'=>'f009'],
            ['name'=>'th','code'=>'f00a'],
            ['name'=>'th-list','code'=>'f00b'],
            ['name'=>'check','code'=>'f00c'],
            ['name'=>'times','code'=>'f00d'],
            ['name'=>'search-plus','code'=>'f00e'],
            ['name'=>'search-minus','code'=>'f010'],
            ['name'=>'power-off','code'=>'f011'],
            ['name'=>'signal','code'=>'f012'],
            ['name'=>'cog','code'=>'f013'],
            ['name'=>'trash-o','code'=>'f014'],
            ['name'=>'home','code'=>'f015'],
            ['name'=>'file-o','code'=>'f016'],
            ['name'=>'clock-o','code'=>'f017'],
            ['name'=>'road','code'=>'f018'],
            ['name'=>'download','code'=>'f019'],
            ['name'=>'arrow-circle-o-down','code'=>'f01a'],
            ['name'=>'arrow-circle-o-up','code'=>'f01b'],
            ['name'=>'inbox','code'=>'f01c'],
            ['name'=>'play-circle-o','code'=>'f01d'],
            ['name'=>'repeat','code'=>'f01e'],
            ['name'=>'refresh','code'=>'f021'],
            ['name'=>'list-alt','code'=>'f022'],
            ['name'=>'lock','code'=>'f023'],
            ['name'=>'flag','code'=>'f024'],
            ['name'=>'headphones','code'=>'f025'],
            ['name'=>'volume-off','code'=>'f026'],
            ['name'=>'volume-down','code'=>'f027'],
            ['name'=>'volume-up','code'=>'f028'],
            ['name'=>'qrcode','code'=>'f029'],
            ['name'=>'barcode','code'=>'f02a'],
            ['name'=>'tag','code'=>'f02b'],
            ['name'=>'tags','code'=>'f02c'],
            ['name'=>'book','code'=>'f02d'],
            ['name'=>'bookmark','code'=>'f02e'],
            ['name'=>'print','code'=>'f02f'],
            ['name'=>'camera','code'=>'f030'],
            ['name'=>'font','code'=>'f031'],
            ['name'=>'bold','code'=>'f032'],
            ['name'=>'italic','code'=>'f033'],
            ['name'=>'text-height','code'=>'f034'],
            ['name'=>'text-width','code'=>'f035'],
            ['name'=>'align-left','code'=>'f036'],
            ['name'=>'align-center','code'=>'f037'],
            ['name'=>'align-right','code'=>'f038'],
            ['name'=>'align-justify','code'=>'f039'],
            ['name'=>'list','code'=>'f03a'],
            ['name'=>'outdent','code'=>'f03b'],
            ['name'=>'indent','code'=>'f03c'],
            ['name'=>'video-camera','code'=>'f03d'],
            ['name'=>'picture-o','code'=>'f03e'],
            ['name'=>'pencil','code'=>'f040'],
            ['name'=>'map-marker','code'=>'f041'],
            ['name'=>'adjust','code'=>'f042'],
            ['name'=>'tint','code'=>'f043'],
            ['name'=>'pencil-square-o','code'=>'f044'],
            ['name'=>'share-square-o','code'=>'f045'],
            ['name'=>'check-square-o','code'=>'f046'],
            ['name'=>'arrows','code'=>'f047'],
            ['name'=>'step-backward','code'=>'f048'],
            ['name'=>'fast-backward','code'=>'f049'],
            ['name'=>'backward','code'=>'f04a'],
            ['name'=>'play','code'=>'f04b'],
            ['name'=>'pause','code'=>'f04c'],
            ['name'=>'stop','code'=>'f04d'],
            ['name'=>'forward','code'=>'f04e'],
            ['name'=>'fast-forward','code'=>'f050'],
            ['name'=>'step-forward','code'=>'f051'],
            ['name'=>'eject','code'=>'f052'],
            ['name'=>'chevron-left','code'=>'f053'],
            ['name'=>'chevron-right','code'=>'f054'],
            ['name'=>'plus-circle','code'=>'f055'],
            ['name'=>'minus-circle','code'=>'f056'],
            ['name'=>'times-circle','code'=>'f057'],
            ['name'=>'check-circle','code'=>'f058'],
            ['name'=>'question-circle','code'=>'f059'],
            ['name'=>'info-circle','code'=>'f05a'],
            ['name'=>'crosshairs','code'=>'f05b'],
            ['name'=>'times-circle-o','code'=>'f05c'],
            ['name'=>'check-circle-o','code'=>'f05d'],
            ['name'=>'ban','code'=>'f05e'],
            ['name'=>'arrow-left','code'=>'f060'],
            ['name'=>'arrow-right','code'=>'f061'],
            ['name'=>'arrow-up','code'=>'f062'],
            ['name'=>'arrow-down','code'=>'f063'],
            ['name'=>'share','code'=>'f064'],
            ['name'=>'expand','code'=>'f065'],
            ['name'=>'compress','code'=>'f066'],
            ['name'=>'plus','code'=>'f067'],
            ['name'=>'minus','code'=>'f068'],
            ['name'=>'asterisk','code'=>'f069'],
            ['name'=>'exclamation-circle','code'=>'f06a'],
            ['name'=>'gift','code'=>'f06b'],
            ['name'=>'leaf','code'=>'f06c'],
            ['name'=>'fire','code'=>'f06d'],
            ['name'=>'eye','code'=>'f06e'],
            ['name'=>'eye-slash','code'=>'f070'],
            ['name'=>'exclamation-triangle','code'=>'f071'],
            ['name'=>'plane','code'=>'f072'],
            ['name'=>'calendar','code'=>'f073'],
            ['name'=>'random','code'=>'f074'],
            ['name'=>'comment','code'=>'f075'],
            ['name'=>'magnet','code'=>'f076'],
            ['name'=>'chevron-up','code'=>'f077'],
            ['name'=>'chevron-down','code'=>'f078'],
            ['name'=>'retweet','code'=>'f079'],
            ['name'=>'shopping-cart','code'=>'f07a'],
            ['name'=>'folder','code'=>'f07b'],
            ['name'=>'folder-open','code'=>'f07c'],
            ['name'=>'arrows-v','code'=>'f07d'],
            ['name'=>'arrows-h','code'=>'f07e'],
            ['name'=>'bar-chart','code'=>'f080'],
            ['name'=>'twitter-square','code'=>'f081'],
            ['name'=>'facebook-square','code'=>'f082'],
            ['name'=>'camera-retro','code'=>'f083'],
            ['name'=>'key','code'=>'f084'],
            ['name'=>'cogs','code'=>'f085'],
            ['name'=>'comments','code'=>'f086'],
            ['name'=>'thumbs-o-up','code'=>'f087'],
            ['name'=>'thumbs-o-down','code'=>'f088'],
            ['name'=>'star-half','code'=>'f089'],
            ['name'=>'heart-o','code'=>'f08a'],
            ['name'=>'sign-out','code'=>'f08b'],
            ['name'=>'linkedin-square','code'=>'f08c'],
            ['name'=>'thumb-tack','code'=>'f08d'],
            ['name'=>'external-link','code'=>'f08e'],
            ['name'=>'sign-in','code'=>'f090'],
            ['name'=>'trophy','code'=>'f091'],
            ['name'=>'github-square','code'=>'f092'],
            ['name'=>'upload','code'=>'f093'],
            ['name'=>'lemon-o','code'=>'f094'],
            ['name'=>'phone','code'=>'f095'],
            ['name'=>'square-o','code'=>'f096'],
            ['name'=>'bookmark-o','code'=>'f097'],
            ['name'=>'phone-square','code'=>'f098'],
            ['name'=>'twitter','code'=>'f099'],
            ['name'=>'facebook','code'=>'f09a'],
            ['name'=>'github','code'=>'f09b'],
            ['name'=>'unlock','code'=>'f09c'],
            ['name'=>'credit-card','code'=>'f09d'],
            ['name'=>'rss','code'=>'f09e'],
            ['name'=>'hdd-o','code'=>'f0a0'],
            ['name'=>'bullhorn','code'=>'f0a1'],
            ['name'=>'bell','code'=>'f0f3'],
            ['name'=>'certificate','code'=>'f0a3'],
            ['name'=>'hand-o-right','code'=>'f0a4'],
            ['name'=>'hand-o-left','code'=>'f0a5'],
            ['name'=>'hand-o-up','code'=>'f0a6'],
            ['name'=>'hand-o-down','code'=>'f0a7'],
            ['name'=>'arrow-circle-left','code'=>'f0a8'],
            ['name'=>'arrow-circle-right','code'=>'f0a9'],
            ['name'=>'arrow-circle-up','code'=>'f0aa'],
            ['name'=>'arrow-circle-down','code'=>'f0ab'],
            ['name'=>'globe','code'=>'f0ac'],
            ['name'=>'wrench','code'=>'f0ad'],
            ['name'=>'tasks','code'=>'f0ae'],
            ['name'=>'filter','code'=>'f0b0'],
            ['name'=>'briefcase','code'=>'f0b1'],
            ['name'=>'arrows-alt','code'=>'f0b2'],
            ['name'=>'users','code'=>'f0c0'],
            ['name'=>'link','code'=>'f0c1'],
            ['name'=>'cloud','code'=>'f0c2'],
            ['name'=>'flask','code'=>'f0c3'],
            ['name'=>'scissors','code'=>'f0c4'],
            ['name'=>'files-o','code'=>'f0c5'],
            ['name'=>'paperclip','code'=>'f0c6'],
            ['name'=>'floppy-o','code'=>'f0c7'],
            ['name'=>'square','code'=>'f0c8'],
            ['name'=>'bars','code'=>'f0c9'],
            ['name'=>'list-ul','code'=>'f0ca'],
            ['name'=>'list-ol','code'=>'f0cb'],
            ['name'=>'strikethrough','code'=>'f0cc'],
            ['name'=>'underline','code'=>'f0cd'],
            ['name'=>'table','code'=>'f0ce'],
            ['name'=>'magic','code'=>'f0d0'],
            ['name'=>'truck','code'=>'f0d1'],
            ['name'=>'pinterest','code'=>'f0d2'],
            ['name'=>'pinterest-square','code'=>'f0d3'],
            ['name'=>'google-plus-square','code'=>'f0d4'],
            ['name'=>'google-plus','code'=>'f0d5'],
            ['name'=>'money','code'=>'f0d6'],
            ['name'=>'caret-down','code'=>'f0d7'],
            ['name'=>'caret-up','code'=>'f0d8'],
            ['name'=>'caret-left','code'=>'f0d9'],
            ['name'=>'caret-right','code'=>'f0da'],
            ['name'=>'columns','code'=>'f0db'],
            ['name'=>'sort','code'=>'f0dc'],
            ['name'=>'sort-desc','code'=>'f0dd'],
            ['name'=>'sort-asc','code'=>'f0de'],
            ['name'=>'envelope','code'=>'f0e0'],
            ['name'=>'linkedin','code'=>'f0e1'],
            ['name'=>'undo','code'=>'f0e2'],
            ['name'=>'gavel','code'=>'f0e3'],
            ['name'=>'tachometer','code'=>'f0e4'],
            ['name'=>'comment-o','code'=>'f0e5'],
            ['name'=>'comments-o','code'=>'f0e6'],
            ['name'=>'bolt','code'=>'f0e7'],
            ['name'=>'sitemap','code'=>'f0e8'],
            ['name'=>'umbrella','code'=>'f0e9'],
            ['name'=>'clipboard','code'=>'f0ea'],
            ['name'=>'lightbulb-o','code'=>'f0eb'],
            ['name'=>'exchange','code'=>'f0ec'],
            ['name'=>'cloud-download','code'=>'f0ed'],
            ['name'=>'cloud-upload','code'=>'f0ee'],
            ['name'=>'user-md','code'=>'f0f0'],
            ['name'=>'stethoscope','code'=>'f0f1'],
            ['name'=>'suitcase','code'=>'f0f2'],
            ['name'=>'bell-o','code'=>'f0a2'],
            ['name'=>'coffee','code'=>'f0f4'],
            ['name'=>'cutlery','code'=>'f0f5'],
            ['name'=>'file-text-o','code'=>'f0f6'],
            ['name'=>'building-o','code'=>'f0f7'],
            ['name'=>'hospital-o','code'=>'f0f8'],
            ['name'=>'ambulance','code'=>'f0f9'],
            ['name'=>'medkit','code'=>'f0fa'],
            ['name'=>'fighter-jet','code'=>'f0fb'],
            ['name'=>'beer','code'=>'f0fc'],
            ['name'=>'h-square','code'=>'f0fd'],
            ['name'=>'plus-square','code'=>'f0fe'],
            ['name'=>'angle-double-left','code'=>'f100'],
            ['name'=>'angle-double-right','code'=>'f101'],
            ['name'=>'angle-double-up','code'=>'f102'],
            ['name'=>'angle-double-down','code'=>'f103'],
            ['name'=>'angle-left','code'=>'f104'],
            ['name'=>'angle-right','code'=>'f105'],
            ['name'=>'angle-up','code'=>'f106'],
            ['name'=>'angle-down','code'=>'f107'],
            ['name'=>'desktop','code'=>'f108'],
            ['name'=>'laptop','code'=>'f109'],
            ['name'=>'tablet','code'=>'f10a'],
            ['name'=>'mobile','code'=>'f10b'],
            ['name'=>'circle-o','code'=>'f10c'],
            ['name'=>'quote-left','code'=>'f10d'],
            ['name'=>'quote-right','code'=>'f10e'],
            ['name'=>'spinner','code'=>'f110'],
            ['name'=>'circle','code'=>'f111'],
            ['name'=>'reply','code'=>'f112'],
            ['name'=>'github-alt','code'=>'f113'],
            ['name'=>'folder-o','code'=>'f114'],
            ['name'=>'folder-open-o','code'=>'f115'],
            ['name'=>'smile-o','code'=>'f118'],
            ['name'=>'frown-o','code'=>'f119'],
            ['name'=>'meh-o','code'=>'f11a'],
            ['name'=>'gamepad','code'=>'f11b'],
            ['name'=>'keyboard-o','code'=>'f11c'],
            ['name'=>'flag-o','code'=>'f11d'],
            ['name'=>'flag-checkered','code'=>'f11e'],
            ['name'=>'terminal','code'=>'f120'],
            ['name'=>'code','code'=>'f121'],
            ['name'=>'reply-all','code'=>'f122'],
            ['name'=>'star-half-o','code'=>'f123'],
            ['name'=>'location-arrow','code'=>'f124'],
            ['name'=>'crop','code'=>'f125'],
            ['name'=>'code-fork','code'=>'f126'],
            ['name'=>'chain-broken','code'=>'f127'],
            ['name'=>'question','code'=>'f128'],
            ['name'=>'info','code'=>'f129'],
            ['name'=>'exclamation','code'=>'f12a'],
            ['name'=>'superscript','code'=>'f12b'],
            ['name'=>'subscript','code'=>'f12c'],
            ['name'=>'eraser','code'=>'f12d'],
            ['name'=>'puzzle-piece','code'=>'f12e'],
            ['name'=>'microphone','code'=>'f130'],
            ['name'=>'microphone-slash','code'=>'f131'],
            ['name'=>'shield','code'=>'f132'],
            ['name'=>'calendar-o','code'=>'f133'],
            ['name'=>'fire-extinguisher','code'=>'f134'],
            ['name'=>'rocket','code'=>'f135'],
            ['name'=>'maxcdn','code'=>'f136'],
            ['name'=>'chevron-circle-left','code'=>'f137'],
            ['name'=>'chevron-circle-right','code'=>'f138'],
            ['name'=>'chevron-circle-up','code'=>'f139'],
            ['name'=>'chevron-circle-down','code'=>'f13a'],
            ['name'=>'html5','code'=>'f13b'],
            ['name'=>'css3','code'=>'f13c'],
            ['name'=>'anchor','code'=>'f13d'],
            ['name'=>'unlock-alt','code'=>'f13e'],
            ['name'=>'bullseye','code'=>'f140'],
            ['name'=>'ellipsis-h','code'=>'f141'],
            ['name'=>'ellipsis-v','code'=>'f142'],
            ['name'=>'rss-square','code'=>'f143'],
            ['name'=>'play-circle','code'=>'f144'],
            ['name'=>'ticket','code'=>'f145'],
            ['name'=>'minus-square','code'=>'f146'],
            ['name'=>'minus-square-o','code'=>'f147'],
            ['name'=>'level-up','code'=>'f148'],
            ['name'=>'level-down','code'=>'f149'],
            ['name'=>'check-square','code'=>'f14a'],
            ['name'=>'pencil-square','code'=>'f14b'],
            ['name'=>'external-link-square','code'=>'f14c'],
            ['name'=>'share-square','code'=>'f14d'],
            ['name'=>'compass','code'=>'f14e'],
            ['name'=>'caret-square-o-down','code'=>'f150'],
            ['name'=>'caret-square-o-up','code'=>'f151'],
            ['name'=>'caret-square-o-right','code'=>'f152'],
            ['name'=>'eur','code'=>'f153'],
            ['name'=>'gbp','code'=>'f154'],
            ['name'=>'usd','code'=>'f155'],
            ['name'=>'inr','code'=>'f156'],
            ['name'=>'jpy','code'=>'f157'],
            ['name'=>'rub','code'=>'f158'],
            ['name'=>'krw','code'=>'f159'],
            ['name'=>'btc','code'=>'f15a'],
            ['name'=>'file','code'=>'f15b'],
            ['name'=>'file-text','code'=>'f15c'],
            ['name'=>'sort-alpha-asc','code'=>'f15d'],
            ['name'=>'sort-alpha-desc','code'=>'f15e'],
            ['name'=>'sort-amount-asc','code'=>'f160'],
            ['name'=>'sort-amount-desc','code'=>'f161'],
            ['name'=>'sort-numeric-asc','code'=>'f162'],
            ['name'=>'sort-numeric-desc','code'=>'f163'],
            ['name'=>'thumbs-up','code'=>'f164'],
            ['name'=>'thumbs-down','code'=>'f165'],
            ['name'=>'youtube-square','code'=>'f166'],
            ['name'=>'youtube','code'=>'f167'],
            ['name'=>'xing','code'=>'f168'],
            ['name'=>'xing-square','code'=>'f169'],
            ['name'=>'youtube-play','code'=>'f16a'],
            ['name'=>'dropbox','code'=>'f16b'],
            ['name'=>'stack-overflow','code'=>'f16c'],
            ['name'=>'instagram','code'=>'f16d'],
            ['name'=>'flickr','code'=>'f16e'],
            ['name'=>'adn','code'=>'f170'],
            ['name'=>'bitbucket','code'=>'f171'],
            ['name'=>'bitbucket-square','code'=>'f172'],
            ['name'=>'tumblr','code'=>'f173'],
            ['name'=>'tumblr-square','code'=>'f174'],
            ['name'=>'long-arrow-down','code'=>'f175'],
            ['name'=>'long-arrow-up','code'=>'f176'],
            ['name'=>'long-arrow-left','code'=>'f177'],
            ['name'=>'long-arrow-right','code'=>'f178'],
            ['name'=>'apple','code'=>'f179'],
            ['name'=>'windows','code'=>'f17a'],
            ['name'=>'android','code'=>'f17b'],
            ['name'=>'linux','code'=>'f17c'],
            ['name'=>'dribbble','code'=>'f17d'],
            ['name'=>'skype','code'=>'f17e'],
            ['name'=>'foursquare','code'=>'f180'],
            ['name'=>'trello','code'=>'f181'],
            ['name'=>'female','code'=>'f182'],
            ['name'=>'male','code'=>'f183'],
            ['name'=>'gratipay','code'=>'f184'],
            ['name'=>'sun-o','code'=>'f185'],
            ['name'=>'moon-o','code'=>'f186'],
            ['name'=>'archive','code'=>'f187'],
            ['name'=>'bug','code'=>'f188'],
            ['name'=>'vk','code'=>'f189'],
            ['name'=>'weibo','code'=>'f18a'],
            ['name'=>'renren','code'=>'f18b'],
            ['name'=>'pagelines','code'=>'f18c'],
            ['name'=>'stack-exchange','code'=>'f18d'],
            ['name'=>'arrow-circle-o-right','code'=>'f18e'],
            ['name'=>'arrow-circle-o-left','code'=>'f190'],
            ['name'=>'caret-square-o-left','code'=>'f191'],
            ['name'=>'dot-circle-o','code'=>'f192'],
            ['name'=>'wheelchair','code'=>'f193'],
            ['name'=>'vimeo-square','code'=>'f194'],
            ['name'=>'try','code'=>'f195'],
            ['name'=>'plus-square-o','code'=>'f196'],
            ['name'=>'space-shuttle','code'=>'f197'],
            ['name'=>'slack','code'=>'f198'],
            ['name'=>'envelope-square','code'=>'f199'],
            ['name'=>'wordpress','code'=>'f19a'],
            ['name'=>'openid','code'=>'f19b'],
            ['name'=>'university','code'=>'f19c'],
            ['name'=>'graduation-cap','code'=>'f19d'],
            ['name'=>'yahoo','code'=>'f19e'],
            ['name'=>'google','code'=>'f1a0'],
            ['name'=>'reddit','code'=>'f1a1'],
            ['name'=>'reddit-square','code'=>'f1a2'],
            ['name'=>'stumbleupon-circle','code'=>'f1a3'],
            ['name'=>'stumbleupon','code'=>'f1a4'],
            ['name'=>'delicious','code'=>'f1a5'],
            ['name'=>'digg','code'=>'f1a6'],
            ['name'=>'pied-piper','code'=>'f1a7'],
            ['name'=>'pied-piper-alt','code'=>'f1a8'],
            ['name'=>'drupal','code'=>'f1a9'],
            ['name'=>'joomla','code'=>'f1aa'],
            ['name'=>'language','code'=>'f1ab'],
            ['name'=>'fax','code'=>'f1ac'],
            ['name'=>'building','code'=>'f1ad'],
            ['name'=>'child','code'=>'f1ae'],
            ['name'=>'paw','code'=>'f1b0'],
            ['name'=>'spoon','code'=>'f1b1'],
            ['name'=>'cube','code'=>'f1b2'],
            ['name'=>'cubes','code'=>'f1b3'],
            ['name'=>'behance','code'=>'f1b4'],
            ['name'=>'behance-square','code'=>'f1b5'],
            ['name'=>'steam','code'=>'f1b6'],
            ['name'=>'steam-square','code'=>'f1b7'],
            ['name'=>'recycle','code'=>'f1b8'],
            ['name'=>'car','code'=>'f1b9'],
            ['name'=>'taxi','code'=>'f1ba'],
            ['name'=>'tree','code'=>'f1bb'],
            ['name'=>'spotify','code'=>'f1bc'],
            ['name'=>'deviantart','code'=>'f1bd'],
            ['name'=>'soundcloud','code'=>'f1be'],
            ['name'=>'database','code'=>'f1c0'],
            ['name'=>'file-pdf-o','code'=>'f1c1'],
            ['name'=>'file-word-o','code'=>'f1c2'],
            ['name'=>'file-excel-o','code'=>'f1c3'],
            ['name'=>'file-powerpoint-o','code'=>'f1c4'],
            ['name'=>'file-image-o','code'=>'f1c5'],
            ['name'=>'file-archive-o','code'=>'f1c6'],
            ['name'=>'file-audio-o','code'=>'f1c7'],
            ['name'=>'file-video-o','code'=>'f1c8'],
            ['name'=>'file-code-o','code'=>'f1c9'],
            ['name'=>'vine','code'=>'f1ca'],
            ['name'=>'codepen','code'=>'f1cb'],
            ['name'=>'jsfiddle','code'=>'f1cc'],
            ['name'=>'life-ring','code'=>'f1cd'],
            ['name'=>'circle-o-notch','code'=>'f1ce'],
            ['name'=>'rebel','code'=>'f1d0'],
            ['name'=>'empire','code'=>'f1d1'],
            ['name'=>'git-square','code'=>'f1d2'],
            ['name'=>'git','code'=>'f1d3'],
            ['name'=>'hacker-news','code'=>'f1d4'],
            ['name'=>'tencent-weibo','code'=>'f1d5'],
            ['name'=>'qq','code'=>'f1d6'],
            ['name'=>'weixin','code'=>'f1d7'],
            ['name'=>'paper-plane','code'=>'f1d8'],
            ['name'=>'paper-plane-o','code'=>'f1d9'],
            ['name'=>'history','code'=>'f1da'],
            ['name'=>'circle-thin','code'=>'f1db'],
            ['name'=>'header','code'=>'f1dc'],
            ['name'=>'paragraph','code'=>'f1dd'],
            ['name'=>'sliders','code'=>'f1de'],
            ['name'=>'share-alt','code'=>'f1e0'],
            ['name'=>'share-alt-square','code'=>'f1e1'],
            ['name'=>'bomb','code'=>'f1e2'],
            ['name'=>'futbol-o','code'=>'f1e3'],
            ['name'=>'tty','code'=>'f1e4'],
            ['name'=>'binoculars','code'=>'f1e5'],
            ['name'=>'plug','code'=>'f1e6'],
            ['name'=>'slideshare','code'=>'f1e7'],
            ['name'=>'twitch','code'=>'f1e8'],
            ['name'=>'yelp','code'=>'f1e9'],
            ['name'=>'newspaper-o','code'=>'f1ea'],
            ['name'=>'wifi','code'=>'f1eb'],
            ['name'=>'calculator','code'=>'f1ec'],
            ['name'=>'paypal','code'=>'f1ed'],
            ['name'=>'google-wallet','code'=>'f1ee'],
            ['name'=>'cc-visa','code'=>'f1f0'],
            ['name'=>'cc-mastercard','code'=>'f1f1'],
            ['name'=>'cc-discover','code'=>'f1f2'],
            ['name'=>'cc-amex','code'=>'f1f3'],
            ['name'=>'cc-paypal','code'=>'f1f4'],
            ['name'=>'cc-stripe','code'=>'f1f5'],
            ['name'=>'bell-slash','code'=>'f1f6'],
            ['name'=>'bell-slash-o','code'=>'f1f7'],
            ['name'=>'trash','code'=>'f1f8'],
            ['name'=>'copyright','code'=>'f1f9'],
            ['name'=>'at','code'=>'f1fa'],
            ['name'=>'eyedropper','code'=>'f1fb'],
            ['name'=>'paint-brush','code'=>'f1fc'],
            ['name'=>'birthday-cake','code'=>'f1fd'],
            ['name'=>'area-chart','code'=>'f1fe'],
            ['name'=>'pie-chart','code'=>'f200'],
            ['name'=>'line-chart','code'=>'f201'],
            ['name'=>'lastfm','code'=>'f202'],
            ['name'=>'lastfm-square','code'=>'f203'],
            ['name'=>'toggle-off','code'=>'f204'],
            ['name'=>'toggle-on','code'=>'f205'],
            ['name'=>'bicycle','code'=>'f206'],
            ['name'=>'bus','code'=>'f207'],
            ['name'=>'ioxhost','code'=>'f208'],
            ['name'=>'angellist','code'=>'f209'],
            ['name'=>'cc','code'=>'f20a'],
            ['name'=>'ils','code'=>'f20b'],
            ['name'=>'meanpath','code'=>'f20c'],
            ['name'=>'buysellads','code'=>'f20d'],
            ['name'=>'connectdevelop','code'=>'f20e'],
            ['name'=>'dashcube','code'=>'f210'],
            ['name'=>'forumbee','code'=>'f211'],
            ['name'=>'leanpub','code'=>'f212'],
            ['name'=>'sellsy','code'=>'f213'],
            ['name'=>'shirtsinbulk','code'=>'f214'],
            ['name'=>'simplybuilt','code'=>'f215'],
            ['name'=>'skyatlas','code'=>'f216'],
            ['name'=>'cart-plus','code'=>'f217'],
            ['name'=>'cart-arrow-down','code'=>'f218'],
            ['name'=>'diamond','code'=>'f219'],
            ['name'=>'ship','code'=>'f21a'],
            ['name'=>'user-secret','code'=>'f21b'],
            ['name'=>'motorcycle','code'=>'f21c'],
            ['name'=>'street-view','code'=>'f21d'],
            ['name'=>'heartbeat','code'=>'f21e'],
            ['name'=>'venus','code'=>'f221'],
            ['name'=>'mars','code'=>'f222'],
            ['name'=>'mercury','code'=>'f223'],
            ['name'=>'transgender','code'=>'f224'],
            ['name'=>'transgender-alt','code'=>'f225'],
            ['name'=>'venus-double','code'=>'f226'],
            ['name'=>'mars-double','code'=>'f227'],
            ['name'=>'venus-mars','code'=>'f228'],
            ['name'=>'mars-stroke','code'=>'f229'],
            ['name'=>'mars-stroke-v','code'=>'f22a'],
            ['name'=>'mars-stroke-h','code'=>'f22b'],
            ['name'=>'neuter','code'=>'f22c'],
            ['name'=>'genderless','code'=>'f22d'],
            ['name'=>'facebook-official','code'=>'f230'],
            ['name'=>'pinterest-p','code'=>'f231'],
            ['name'=>'whatsapp','code'=>'f232'],
            ['name'=>'server','code'=>'f233'],
            ['name'=>'user-plus','code'=>'f234'],
            ['name'=>'user-times','code'=>'f235'],
            ['name'=>'bed','code'=>'f236'],
            ['name'=>'viacoin','code'=>'f237'],
            ['name'=>'train','code'=>'f238'],
            ['name'=>'subway','code'=>'f239'],
            ['name'=>'medium','code'=>'f23a'],
            ['name'=>'y-combinator','code'=>'f23b'],
            ['name'=>'optin-monster','code'=>'f23c'],
            ['name'=>'opencart','code'=>'f23d'],
            ['name'=>'expeditedssl','code'=>'f23e'],
            ['name'=>'battery-full','code'=>'f240'],
            ['name'=>'battery-three-quarters','code'=>'f241'],
            ['name'=>'battery-half','code'=>'f242'],
            ['name'=>'battery-quarter','code'=>'f243'],
            ['name'=>'battery-empty','code'=>'f244'],
            ['name'=>'mouse-pointer','code'=>'f245'],
            ['name'=>'i-cursor','code'=>'f246'],
            ['name'=>'object-group','code'=>'f247'],
            ['name'=>'object-ungroup','code'=>'f248'],
            ['name'=>'sticky-note','code'=>'f249'],
            ['name'=>'sticky-note-o','code'=>'f24a'],
            ['name'=>'cc-jcb','code'=>'f24b'],
            ['name'=>'cc-diners-club','code'=>'f24c'],
            ['name'=>'clone','code'=>'f24d'],
            ['name'=>'balance-scale','code'=>'f24e'],
            ['name'=>'hourglass-o','code'=>'f250'],
            ['name'=>'hourglass-start','code'=>'f251'],
            ['name'=>'hourglass-half','code'=>'f252'],
            ['name'=>'hourglass-end','code'=>'f253'],
            ['name'=>'hourglass','code'=>'f254'],
            ['name'=>'hand-rock-o','code'=>'f255'],
            ['name'=>'hand-paper-o','code'=>'f256'],
            ['name'=>'hand-scissors-o','code'=>'f257'],
            ['name'=>'hand-lizard-o','code'=>'f258'],
            ['name'=>'hand-spock-o','code'=>'f259'],
            ['name'=>'hand-pointer-o','code'=>'f25a'],
            ['name'=>'hand-peace-o','code'=>'f25b'],
            ['name'=>'trademark','code'=>'f25c'],
            ['name'=>'registered','code'=>'f25d'],
            ['name'=>'creative-commons','code'=>'f25e'],
            ['name'=>'gg','code'=>'f260'],
            ['name'=>'gg-circle','code'=>'f261'],
            ['name'=>'tripadvisor','code'=>'f262'],
            ['name'=>'odnoklassniki','code'=>'f263'],
            ['name'=>'odnoklassniki-square','code'=>'f264'],
            ['name'=>'get-pocket','code'=>'f265'],
            ['name'=>'wikipedia-w','code'=>'f266'],
            ['name'=>'safari','code'=>'f267'],
            ['name'=>'chrome','code'=>'f268'],
            ['name'=>'firefox','code'=>'f269'],
            ['name'=>'opera','code'=>'f26a'],
            ['name'=>'internet-explorer','code'=>'f26b'],
            ['name'=>'television','code'=>'f26c'],
            ['name'=>'contao','code'=>'f26d'],
            ['name'=>'500px','code'=>'f26e'],
            ['name'=>'amazon','code'=>'f270'],
            ['name'=>'calendar-plus-o','code'=>'f271'],
            ['name'=>'calendar-minus-o','code'=>'f272'],
            ['name'=>'calendar-times-o','code'=>'f273'],
            ['name'=>'calendar-check-o','code'=>'f274'],
            ['name'=>'industry','code'=>'f275'],
            ['name'=>'map-pin','code'=>'f276'],
            ['name'=>'map-signs','code'=>'f277'],
            ['name'=>'map-o','code'=>'f278'],
            ['name'=>'map','code'=>'f279'],
            ['name'=>'commenting','code'=>'f27a'],
            ['name'=>'commenting-o','code'=>'f27b'],
            ['name'=>'houzz','code'=>'f27c'],
            ['name'=>'vimeo','code'=>'f27d'],
            ['name'=>'black-tie','code'=>'f27e'],
            ['name'=>'fonticons','code'=>'f280'],
        ];

        foreach ($icons as $icon) {
            $data[$icon['name']] = '&#x' . $icon['code'] . ' ' . ucwords(str_replace('-', ' ', $icon['name']));
        }

        ksort($data);

        return $data;
    }
}
